/*
 * SPDX-FileCopyrightText: 2023-2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "esp_private/esp_crypto_lock_internal.h"
#include "memory_checks.h"
#include "unity_fixture.h"

#include "soc/soc_caps.h"
#include "esp_log.h"
#include "ds_types.h"

ESP_LOG_ATTR_TAG(TAG, "test_ds");

#include "rom/efuse.h"
#include "rom/sha.h"
#include "rom/digital_signature.h"
#include "rom/aes.h"
#include "rom/hmac.h"

#if SOC_KEY_MANAGER_DS_KEY_DEPLOY
#include "hal/key_mgr_ll.h"
#endif

#if CONFIG_IDF_TARGET_ESP32S2
#include "soc/soc_memory_layout.h"
#endif

#define NUM_RESULTS 10

#if CONFIG_IDF_TARGET_ESP32S2 || CONFIG_IDF_TARGET_ESP32S3
#define DS_MAX_BITS (4096)
#else
#define DS_MAX_BITS (ETS_DS_MAX_BITS)
#endif

typedef struct {
    uint8_t iv[ETS_DS_IV_LEN];
    ets_ds_p_data_t p_data;
    uint8_t expected_c[ETS_DS_C_LEN];
    uint8_t hmac_key_idx;
    uint32_t expected_results[NUM_RESULTS][DS_MAX_BITS / 32];
} encrypt_testcase_t;


#if DS_MAX_BITS == 4096
#define RSA_LEN         (ESP_DS_RSA_4096)
#include "digital_signature_test_cases_4096.h"
#elif DS_MAX_BITS == 3072
#define RSA_LEN         (ESP_DS_RSA_3072)
#include "digital_signature_test_cases_3072.h"
#endif

_Static_assert(NUM_RESULTS == NUM_MESSAGES, "expected_results size should be the same as NUM_MESSAGES in generated header");

#if !CONFIG_IDF_TARGET_ESP32S2

#include "esp_private/periph_ctrl.h"
#include "hal/aes_ll.h"
#include "hal/ds_hal.h"
#include "hal/ds_ll.h"
#include "hal/hmac_hal.h"
#include "hal/hmac_ll.h"
#include "hal/sha_ll.h"
#include "esp_crypto_periph_clk.h"

static void ds_acquire_enable(void)
{
    // We also enable SHA and HMAC here. SHA is used by HMAC, HMAC is used by DS.
    esp_crypto_hmac_enable_periph_clk(true);
    esp_crypto_sha_enable_periph_clk(true);
    esp_crypto_mpi_enable_periph_clk(true);
    esp_crypto_ds_enable_periph_clk(true);
}

static void ds_disable_release(void)
{
    esp_crypto_mpi_enable_periph_clk(false);
    esp_crypto_sha_enable_periph_clk(false);
    esp_crypto_hmac_enable_periph_clk(false);
    esp_crypto_ds_enable_periph_clk(false);
}


static esp_err_t esp_ds_start_sign(const void *message, const esp_ds_data_t *data, uint32_t key_id)
{
    ds_acquire_enable();

#if SOC_KEY_MANAGER_DS_KEY_DEPLOY
    if (key_id == HMAC_KEY_KM) {
        ds_hal_set_key_source(DS_KEY_SOURCE_KEY_MGR);
    } else {
        ds_hal_set_key_source(DS_KEY_SOURCE_EFUSE);
#endif
        hmac_hal_start();
        uint32_t conf_error = hmac_hal_configure(HMAC_OUTPUT_DS, key_id);
        if (conf_error) {
            ds_disable_release();
            ESP_LOGE(TAG, "HMAC configure failed");
            return ESP_ERR_HW_CRYPTO_DS_HMAC_FAIL;
        }
#if SOC_KEY_MANAGER_DS_KEY_DEPLOY
    }
#endif

    ds_hal_start();

    while (ds_hal_busy() != 0) { }

    size_t rsa_len = (data->rsa_length + 1) * 4;
    ds_hal_write_private_key_params(data->c);
    ds_hal_configure_iv((uint32_t *)data->iv);
    ds_hal_write_message(message, rsa_len);

    ds_hal_start_sign();

    return ESP_OK;
}

static esp_err_t esp_ds_finish_sign(void *signature, const esp_ds_data_t *data)
{
    unsigned rsa_len = (data->rsa_length + 1) * 4;

    while (ds_hal_busy()) { }

    ds_signature_check_t sig_check_result = ds_hal_read_result((uint8_t *) signature, (size_t) rsa_len);

    esp_err_t return_value = ESP_OK;

    if (sig_check_result == DS_SIGNATURE_MD_FAIL || sig_check_result == DS_SIGNATURE_PADDING_AND_MD_FAIL) {
        return_value = ESP_ERR_HW_CRYPTO_DS_INVALID_DIGEST;
    }

    if (sig_check_result == DS_SIGNATURE_PADDING_FAIL) {
        return_value = ESP_ERR_HW_CRYPTO_DS_INVALID_PADDING;
    }

    hmac_hal_clean();
    ds_hal_finish();
    ds_disable_release();

    return return_value;
}

esp_err_t esp_ds_sign(const void *message,
                      const esp_ds_data_t *data,
                      uint32_t key_id,
                      void *signature)
{
    esp_err_t result = esp_ds_start_sign(message, data, key_id);
    if (result != ESP_OK) {
        return result;
    }

    while (ds_hal_busy()) { }

    return esp_ds_finish_sign(signature, data);
}

static esp_err_t esp_ds_encrypt_params(esp_ds_data_t *data,
                                const void *iv,
                                const esp_ds_p_data_t *p_data,
                                const void *key)
{
    if (!p_data) {
        return ESP_ERR_INVALID_ARG;
    }

    esp_err_t result = ESP_OK;

    AES_RCC_ATOMIC() {
        aes_ll_enable_bus_clock(true);
        aes_ll_reset_register();
    }

    SHA_RCC_ATOMIC() {
        sha_ll_enable_bus_clock(true);
        sha_ll_reset_register();
    }

    ets_ds_data_t *ds_data = (ets_ds_data_t *) data;
    const ets_ds_p_data_t *ds_plain_data = (const ets_ds_p_data_t *) p_data;

    ets_ds_result_t ets_result = ets_ds_encrypt_params(ds_data, iv, ds_plain_data, key, ETS_DS_KEY_HMAC);

    if (ets_result == ETS_DS_INVALID_PARAM) {
        result = ESP_ERR_INVALID_ARG;
    }

    SHA_RCC_ATOMIC() {
        sha_ll_enable_bus_clock(false);
    }

    AES_RCC_ATOMIC() {
        aes_ll_enable_bus_clock(false);
    }

    return result;
}
#else /* !CONFIG_IDF_TARGET_ESP32S2 */

static void ds_acquire_enable(void)
{
    ets_hmac_enable();
    ets_ds_enable();
}

static void ds_disable_release(void)
{
    ets_ds_disable();
    ets_hmac_disable();
}

static esp_err_t esp_ds_start_sign(const void *message,
                                const esp_ds_data_t *data,
                                uint32_t key_id)
{
    ds_acquire_enable();

    int r = ets_hmac_calculate_downstream(ETS_EFUSE_BLOCK_KEY0 + (ets_efuse_block_t) key_id,
                                          ETS_EFUSE_KEY_PURPOSE_HMAC_DOWN_DIGITAL_SIGNATURE);
    if (r != ETS_OK) {
        ds_disable_release();
        return ESP_ERR_HW_CRYPTO_DS_HMAC_FAIL;
    }

    ets_ds_data_t *ds_data = (ets_ds_data_t *) data;

    ets_ds_result_t result = ets_ds_start_sign(message, ds_data);

    // ETS_DS_INVALID_PARAM only happens if a parameter is NULL or data->rsa_length is wrong
    // We checked all of that already
    assert(result != ETS_DS_INVALID_PARAM);

    if (result == ETS_DS_INVALID_KEY) {
        ds_disable_release();
        return ESP_ERR_HW_CRYPTO_DS_INVALID_KEY;
    }

    return ESP_OK;
}

esp_err_t esp_ds_finish_sign(void *signature, const esp_ds_data_t *data)
{
    ets_ds_result_t result = ets_ds_finish_sign(signature, (ets_ds_data_t*) data);

    esp_err_t return_value = ESP_OK;

    assert(result != ETS_DS_INVALID_PARAM);

    if (result == ETS_DS_INVALID_DIGEST) {
        return_value = ESP_ERR_HW_CRYPTO_DS_INVALID_DIGEST;
    }
    if (result == ETS_DS_INVALID_PADDING) {
        return_value = ESP_ERR_HW_CRYPTO_DS_INVALID_PADDING;
    }

    int res = ets_hmac_invalidate_downstream(ETS_EFUSE_KEY_PURPOSE_HMAC_DOWN_DIGITAL_SIGNATURE);
    assert(res == ETS_OK); // should not fail if called with correct purpose
    (void)res;

    ds_disable_release();

    return return_value;
}

static esp_err_t esp_ds_sign(const void *message,
                      const esp_ds_data_t *data,
                      uint32_t key_id,
                      void *signature)
{
    esp_err_t result = esp_ds_start_sign(message, data, key_id);
    if (result != ESP_OK) {
        return result;
    }

    while (ets_ds_is_busy()) { }

    return esp_ds_finish_sign(signature, (void *)data);
}

static esp_err_t esp_ds_encrypt_params(esp_ds_data_t *data,
                                const void *iv,
                                const esp_ds_p_data_t *p_data,
                                const void *key)
{
    assert(esp_ptr_internal(p_data) && esp_ptr_word_aligned(p_data));

    esp_err_t result = ESP_OK;

    ets_aes_enable();
    ets_sha_enable();

    ets_ds_data_t *ds_data = (ets_ds_data_t *) data;
    const ets_ds_p_data_t *ds_plain_data = (const ets_ds_p_data_t *) p_data;

    ets_ds_result_t ets_result = ets_ds_encrypt_params(ds_data, iv, ds_plain_data, key, ETS_DS_KEY_HMAC);

    if (ets_result == ETS_DS_INVALID_PARAM) {
        result = ESP_ERR_INVALID_ARG;
    }

    ets_sha_disable();
    ets_aes_disable();

    return result;
}

#endif /* !CONFIG_IDF_TARGET_ESP32S2 */

TEST_GROUP(ds);

TEST_SETUP(ds)
{
    test_utils_record_free_mem();
    TEST_ESP_OK(test_utils_set_leak_level(0, ESP_LEAK_TYPE_CRITICAL, ESP_COMP_LEAK_GENERAL));
}

TEST_TEAR_DOWN(ds)
{
    test_utils_finish_and_evaluate_leaks(test_utils_get_leak_level(ESP_LEAK_TYPE_WARNING, ESP_COMP_LEAK_ALL),
                                         test_utils_get_leak_level(ESP_LEAK_TYPE_CRITICAL, ESP_COMP_LEAK_ALL));
}

TEST(ds, digital_signature_parameter_encryption)
{
    for (int i = 0; i < NUM_CASES; i++) {
        ESP_LOGI(TAG, "Encrypting test case %d.", i);
        const encrypt_testcase_t *t = &test_cases[i];
        esp_ds_data_t result = { };
        esp_ds_p_data_t p_data;

        memcpy(p_data.Y, t->p_data.Y, DS_MAX_BITS / 8);
        memcpy(p_data.M, t->p_data.M, DS_MAX_BITS / 8);
        memcpy(p_data.Rb, t->p_data.Rb, DS_MAX_BITS / 8);
        p_data.M_prime = t->p_data.M_prime;
        p_data.length = t->p_data.length;

        esp_err_t r = esp_ds_encrypt_params(&result, t->iv, &p_data,
                                            test_hmac_keys[t->hmac_key_idx]);
        ESP_LOGI(TAG, "Encrypting test case %d done", i);
        TEST_ASSERT_EQUAL(ESP_OK, r);
        TEST_ASSERT_EQUAL(t->p_data.length, result.rsa_length);
        TEST_ASSERT_EQUAL_HEX8_ARRAY(t->iv, result.iv, ETS_DS_IV_LEN);
        TEST_ASSERT_EQUAL_HEX8_ARRAY(t->expected_c, result.c, ETS_DS_C_LEN);
    }
}

// This test uses the HMAC_KEY_BLOCK_1 eFuse key which hasn't been burned by burn_hmac_keys().
// HMAC_KEY_BLOCK_1 is usually used for HMAC upstream (user access) tests.
TEST(ds, digital_signature_wrong_hmac_key_purpose)
{
    esp_ds_data_t ds_data = {};
    ds_data.rsa_length =  RSA_LEN;
    const char *message = "test";

    // HMAC fails in that case because it checks for the correct purpose
    TEST_ASSERT_EQUAL(ESP_ERR_HW_CRYPTO_DS_HMAC_FAIL, esp_ds_start_sign(message, &ds_data, HMAC_KEY_BLOCK_1));
}

// This test uses the HMAC_KEY_BLOCK_1 eFuse key which hasn't been burned by burn_hmac_keys().
// HMAC_KEY_BLOCK_1 is usually used for HMAC upstream (user access) tests.
TEST(ds, digital_signature_blocking_wrong_hmac_key_purpose)
{
    esp_ds_data_t ds_data = {};
    ds_data.rsa_length = RSA_LEN;
    const char *message = "test";
    uint8_t signature_data [128 * 4];

    // HMAC fails in that case because it checks for the correct purpose
    TEST_ASSERT_EQUAL(ESP_ERR_HW_CRYPTO_DS_HMAC_FAIL, esp_ds_sign(message, &ds_data, HMAC_KEY_BLOCK_1, signature_data));
}

TEST(ds, digital_signature_operation)
{
    for (int i = 0; i < NUM_CASES; i++) {
        ESP_LOGI(TAG, "Running test case %d.", i);
        const encrypt_testcase_t *t = &test_cases[i];

        // copy encrypt parameter test case into ds_data structure
        esp_ds_data_t ds_data = { };
        memcpy(ds_data.iv, t->iv, ETS_DS_IV_LEN);
        memcpy(ds_data.c, t->expected_c, ETS_DS_C_LEN);
        ds_data.rsa_length = t->p_data.length;

        for (int j = 0; j < NUM_MESSAGES; j++) {
            uint8_t signature[DS_MAX_BITS / 8] = { 0 };
            ESP_LOGD(TAG, " ... message %d", j);

            esp_err_t ds_r = esp_ds_start_sign(test_messages[j],
                                               &ds_data,
                                               t->hmac_key_idx);
            TEST_ASSERT_EQUAL(ESP_OK, ds_r);

            ds_r = esp_ds_finish_sign(signature, &ds_data);
            TEST_ASSERT_EQUAL(ESP_OK, ds_r);

            TEST_ASSERT_EQUAL_HEX8_ARRAY(t->expected_results[j], signature, sizeof(signature));
        }
#if CONFIG_IDF_TARGET_ESP32C3
        ets_hmac_invalidate_downstream(ETS_EFUSE_KEY_PURPOSE_HMAC_DOWN_DIGITAL_SIGNATURE);
#endif
    }
}

TEST(ds, digital_signature_blocking_operation)
{
    for (int i = 0; i < NUM_CASES; i++) {
        ESP_LOGI(TAG, "Running test case %d.", i);
        const encrypt_testcase_t *t = &test_cases[i];

        // copy encrypt parameter test case into ds_data structure
        esp_ds_data_t ds_data = { };
        memcpy(ds_data.iv, t->iv, ETS_DS_IV_LEN);
        memcpy(ds_data.c, t->expected_c, ETS_DS_C_LEN);
        ds_data.rsa_length = t->p_data.length;

        uint8_t signature[DS_MAX_BITS / 8] = { 0 };
#if CONFIG_IDF_TARGET_ESP32S2 || CONFIG_IDF_TARGET_ESP32S3

        esp_err_t ds_r = esp_ds_start_sign(test_messages[0],
                                           &ds_data,
                                           t->hmac_key_idx);
        TEST_ASSERT_EQUAL(ESP_OK, ds_r);

        ds_r = esp_ds_finish_sign(signature, &ds_data);
        TEST_ASSERT_EQUAL(ESP_OK, ds_r);
#else
        esp_err_t ds_r = esp_ds_sign(test_messages[0],
                                     &ds_data,
                                     t->hmac_key_idx,
                                     signature);
        TEST_ASSERT_EQUAL(ESP_OK, ds_r);
#endif

        TEST_ASSERT_EQUAL_HEX8_ARRAY(t->expected_results[0], signature, sizeof(signature));
    }
}

TEST(ds, digital_signature_invalid_data)
{
    // Set up a valid test case
    const encrypt_testcase_t *t = &test_cases[0];
    esp_ds_data_t ds_data = { };
    memcpy(ds_data.iv, t->iv, ETS_DS_IV_LEN);
    memcpy(ds_data.c, t->expected_c, ETS_DS_C_LEN);
    ds_data.rsa_length = t->p_data.length;

    uint8_t signature[DS_MAX_BITS / 8] = { 0 };
    const uint8_t zero[DS_MAX_BITS / 8] = { 0 };

    // Corrupt the IV one bit at a time, rerun and expect failure
    for (int bit = 0; bit < 128; bit++) {
        ESP_LOGD(TAG, "Corrupting IV bit %d.", bit);
        ds_data.iv[bit / 8] ^= 1 << (bit % 8);

        esp_err_t ds_r = esp_ds_start_sign(test_messages[0], &ds_data, t->hmac_key_idx);
        TEST_ASSERT_EQUAL(ESP_OK, ds_r);

        ds_r = esp_ds_finish_sign(signature, &ds_data);
        TEST_ASSERT_EQUAL(ESP_ERR_HW_CRYPTO_DS_INVALID_DIGEST, ds_r);

        TEST_ASSERT_EQUAL_HEX8_ARRAY(zero, signature, DS_MAX_BITS / 8);
        ds_data.iv[bit / 8] ^= 1 << (bit % 8);
    }

    // Corrupt encrypted key data one bit at a time, rerun and expect failure
    ESP_LOGD(TAG, "Corrupting C.");
    for (int bit = 0; bit < ETS_DS_C_LEN * 8; bit++) {
        ESP_LOGD(TAG, "Corrupting C bit %d.", bit);
        ds_data.c[bit / 8] ^= 1 << (bit % 8);

        esp_err_t ds_r = esp_ds_start_sign(test_messages[0], &ds_data, t->hmac_key_idx);
        TEST_ASSERT_EQUAL(ESP_OK, ds_r);

        ds_r = esp_ds_finish_sign(signature, &ds_data);
        TEST_ASSERT_EQUAL(ESP_ERR_HW_CRYPTO_DS_INVALID_DIGEST, ds_r);

        TEST_ASSERT_EQUAL_HEX8_ARRAY(zero, signature, DS_MAX_BITS / 8);

        ds_data.c[bit / 8] ^= 1 << (bit % 8);
    }
}

TEST_GROUP_RUNNER(ds)
{
    RUN_TEST_CASE(ds, digital_signature_parameter_encryption);
    RUN_TEST_CASE(ds, digital_signature_wrong_hmac_key_purpose);
    RUN_TEST_CASE(ds, digital_signature_blocking_wrong_hmac_key_purpose);
    RUN_TEST_CASE(ds, digital_signature_operation);
    RUN_TEST_CASE(ds, digital_signature_blocking_operation);
    RUN_TEST_CASE(ds, digital_signature_invalid_data);
}
