package io.agora.scene.convoai.ui

import androidx.annotation.IntDef

/**
 * Sealed base class for resource error types.
 * Extend this class to represent specific resource errors (e.g., picture, audio, etc.).
 */
sealed class ResourceError

/**
 * Picture error data class, extends ResourceError.
 * Used to represent errors related to image resources.
 * @property uuid Unique identifier for the image
 * @property success Whether the operation was successful
 * @property errorCode Error code if the operation failed
 * @property errorMessage Error message if the operation failed
 */
data class PictureError(
    val uuid: String,
    val success: Boolean,
    val errorCode: Int?,
    val errorMessage: String?
) : ResourceError()


// Base class for all media information types
sealed class MediaInfo

/**
 * Picture information data class, extends MediaInfo
 * @property uuid Unique identifier for the image
 * @property width Image width in pixels
 * @property height Image height in pixels
 * @property sizeBytes Image file size in bytes
 * @property sourceType Source type of the image (e.g., local, remote)
 * @property sourceValue Source value (e.g., file path or URL)
 * @property uploadTime Upload timestamp in milliseconds
 * @property totalUserImages Total number of user images
 */
data class PictureInfo(
    val uuid: String,
    val width: Int,
    val height: Int,
    val sizeBytes: Long,
    val sourceType: String,
    val sourceValue: String,
    val uploadTime: Long,
    val totalUserImages: Int,
) : MediaInfo()

/**
 * Render mode constants and annotation
 */
@IntDef(CovRenderMode.TEXT, CovRenderMode.WORD, CovRenderMode.SYNC_TEXT)
@Retention(AnnotationRetention.RUNTIME)
@Target(AnnotationTarget.FIELD, AnnotationTarget.PROPERTY, AnnotationTarget.VALUE_PARAMETER)
annotation class CovRenderMode {
    companion object {
        const val WORD = 0 // Word by word:The transcript appears one word at a time, synchronized with the AI's speech.
        const val SYNC_TEXT = 1 // Transcript-Voice sync text mode
        const val TEXT = 2 // Streaming text:The transcript is displayed in chunks as they are generated by the AI.
    }
}

data class CovTranscriptRender(
    @CovRenderMode val renderMode: Int,
    val text: String,
    val detail: String
)